<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use Illuminate\Support\Facades\File;

class CustomInstallerController extends Controller
{
    public function index()
    {
        // Check if already installed
        if (file_exists(storage_path('installed'))) {
            return redirect('/');
        }

        return view('installer.index');
    }

    public function install(Request $request)
    {
        $request->validate([
            'db_host' => 'required',
            'db_port' => 'required',
            'db_database' => 'required',
            'db_username' => 'required',
            'db_password' => 'nullable',
            'admin_name' => 'required',
            'admin_email' => 'required|email',
            'admin_password' => 'required|min:8',
        ]);

        // 1. Test Database Connection
        try {
            $connection = new \PDO(
                "mysql:host={$request->db_host};port={$request->db_port};dbname={$request->db_database}",
                $request->db_username,
                $request->db_password
            );
        } catch (\Exception $e) {
            return back()->withErrors(['db_error' => 'Could not connect to database: ' . $e->getMessage()])->withInput();
        }

        // 2. Write .env file
        $this->writeEnv([
            'DB_HOST' => $request->db_host,
            'DB_PORT' => $request->db_port,
            'DB_DATABASE' => $request->db_database,
            'DB_USERNAME' => $request->db_username,
            'DB_PASSWORD' => $request->db_password,
            'APP_URL' => $request->getSchemeAndHttpHost(),
        ]);

        // 3. Run Migrations
        try {
            // We need to set the config at runtime to run migrations immediately
            config([
                'database.connections.mysql.host' => $request->db_host,
                'database.connections.mysql.port' => $request->db_port,
                'database.connections.mysql.database' => $request->db_database,
                'database.connections.mysql.username' => $request->db_username,
                'database.connections.mysql.password' => $request->db_password,
            ]);
            
            DB::purge('mysql');
            DB::reconnect('mysql');

            Artisan::call('migrate', ['--force' => true]);
            Artisan::call('db:seed', ['--force' => true]); // Optional: Seed default data
            
        } catch (\Exception $e) {
            return back()->withErrors(['migration_error' => 'Migration failed: ' . $e->getMessage()])->withInput();
        }

        // 4. Create Admin User
        try {
            $user = User::create([
                'name' => $request->admin_name,
                'email' => $request->admin_email,
                'password' => Hash::make($request->admin_password),
                'role' => 'admin', // Adjust based on your User model
                'email_verified_at' => now(),
            ]);
        } catch (\Exception $e) {
             // User might already exist if migrations seeded data
             // Try to update existing admin or just proceed
        }

        // 5. Mark as Installed
        file_put_contents(storage_path('installed'), 'INSTALLED on ' . now());

        return redirect()->route('installer.success');
    }

    public function success()
    {
        return view('installer.success');
    }

    private function writeEnv($data)
    {
        $path = base_path('.env');
        
        if (!file_exists($path)) {
            copy(base_path('.env.example'), $path);
        }

        $fileContent = file_get_contents($path);

        foreach ($data as $key => $value) {
            // Quote values if they contain spaces
            if (str_contains($value, ' ')) {
                $value = '"' . $value . '"';
            }

            // Check if key exists
            if (preg_match("/^{$key}=.*/m", $fileContent)) {
                // Replace existing
                $fileContent = preg_replace("/^{$key}=.*/m", "{$key}={$value}", $fileContent);
            } else {
                // Append new
                $fileContent .= "\n{$key}={$value}";
            }
        }

        file_put_contents($path, $fileContent);
    }
}
